unit Misc;

interface

uses Classes, ExtCtrls, Contnrs, PluginSupportUnit, MailToolsUnit;

type
  TUniqueQueue = class(TQueue)
  protected
    procedure PushItem(AItem: Pointer); override;
  end;

  TMessagePriority = (mpHighest, mpHigh, mpNormal, mpLow, mpLowest);

  TMailItem = class(TCollectionItem)
  public
    MsgNum : integer;
    From : string;
    MailTo : string;
    Address : string;
    ReplyTo : string;
    Subject : string;
    Date : string;
    Size : integer;
    Priority : TMessagePriority;
    Ignored : boolean;
    HasAttachment : boolean;
    Viewed : boolean;
    IsInWhiteList : boolean;
    New : boolean;
    Important : boolean;
    Spam : boolean;
    Protect : boolean;
    MsgID : string;
    UID : string;
  end;

  TMailItems = class(TCollection)
  private
    function GetItem(Index: Integer): TMailItem;
    procedure SetItem(Index: Integer; Value: TMailItem);
  protected
    procedure Update(Item: TCollectionItem); override;
  public
    constructor Create;
    function Add: TMailItem;
    function FindMessage(MsgNum: Integer): TMailItem;
    function FindUID(UID: string): TMailItem;
    function FindUIDWithDuplicates(UID: string): TMailItem;
    procedure SetAllMsgNum(Value : integer);
    procedure SetAllNew(Value: boolean);
    procedure DeleteAllMsgNum(MsgNum : integer);
    property Items[Index: Integer]: TMailItem read GetItem write SetItem; default;
  end;

  TAccountItem = class(TCollectionItem)
  public
    Name : string;
    Server : string;
    SMTPServer : string;
    Port : integer;
    SMTPPort: integer;
    Protocol : string;
    Login : string;
    Password : string;
    MailProgram : string;
    Sound : string;
    Color : string;
    MsgIDs : string;
    ViewedMsgIDs : TStringList;
    IgnoreCount : integer;
    Enabled : boolean;
    Error : boolean;
    Status : string;
    Interval : integer;
    Timer : TTimer;
    Size : integer;
    Connecting : boolean;
    OldNum : integer;
    Prot : TPluginProtocol;
    UIDLSupported : boolean;
    Mail : TMailItems;
    UnknownMessages: TUnknownMsg;
  protected
    procedure AssignTo(Dest: TPersistent); override;
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
  end;

  TAccountItems = class(TCollection)
  private
    function GetItem(Index: Integer): TAccountItem;
    procedure SetItem(Index: Integer; Value: TAccountItem);
  public
    constructor Create;
    function Add: TAccountItem;
    procedure Move(CurIndex, NewIndex: Integer);
    property Items[Index: Integer]: TAccountItem read GetItem write SetItem; default;
  end;

  TRuleArea = (raHeader,raFrom,raSubject,raTo,raCC,raFromName,raFromAddress,raBody);
  TRuleCompare = (rcContains,rcEquals,rcWildcard,rcEmpty,rcNotContains);

  TRuleItem = class(TCollectionItem)
  public
    Name : string;
    Enabled : boolean;
    New : boolean;
    Account : integer;
    Area : TRuleArea;
    Compare : TRuleCompare;
    Text : string;
    Wav : string;
    Delete : boolean;
    Ignore : boolean;
    EXE : string;
    Important : boolean;
    Spam : boolean;
    Protect : boolean;
    Log : boolean;
  protected
    procedure AssignTo(Dest: TPersistent); override;
  end;

  TRuleItems = class(TCollection)
  private
    function GetItem(Index: Integer): TRuleItem;
    procedure SetItem(Index: Integer; Value: TRuleItem);
  public
    constructor Create;
    function Add: TRuleItem;
    procedure Move(CurIndex, NewIndex: Integer);
    property Items[Index: Integer]: TRuleItem read GetItem write SetItem; default;
  end;


implementation

{ TUniqueQueue }

procedure TUniqueQueue.PushItem(AItem: Pointer);
begin
  if List.IndexOf(AItem) = -1 then
    inherited;
end;


{ TMailItems }

constructor TMailItems.Create;
begin
  inherited Create(TMailItem);
end;

procedure TMailItems.Update(Item: TCollectionItem);
begin
  inherited;

end;

function TMailItems.GetItem(Index: Integer): TMailItem;
begin
  Result := TMailItem(inherited GetItem(Index));
end;

procedure TMailItems.SetItem(Index: Integer; Value: TMailItem);
begin
  inherited SetItem(Index, Value);
end;

function TMailItems.Add: TMailItem;
begin
  Result := TMailItem(inherited Add);
end;

function TMailItems.FindMessage(MsgNum: Integer): TMailItem;
////////////////////////////////////////////////////////////////////////////////
// Find message with MsgNum
var
  i : Integer;
begin
  for i := 0 to Count-1 do
  begin
    Result := Items[i];
    if Result.MsgNum = MsgNum then Exit;
  end;
  Result := nil;
end;

function TMailItems.FindUID(UID: string): TMailItem;
////////////////////////////////////////////////////////////////////////////////
// Find message with UID
var
  i : Integer;
begin
  for i := 0 to Count-1 do
  begin
    Result := Items[i];
    if Result.UID = UID then Exit;
  end;
  Result := nil;
end;

function TMailItems.FindUIDWithDuplicates(UID: string): TMailItem;
////////////////////////////////////////////////////////////////////////////////
// Find message with UID - non-unique UID allowed according to RFC1939
// NOTE: Call SetAllMsgNum(-1) before using this method
var
  i : Integer;
begin
  for i := 0 to Count-1 do
  begin
    Result := TMailItem(Items[i]);
    if (Result.UID = UID) and (Result.MsgNum = -1) then Exit;
  end;
  Result := nil;
end;

procedure TMailItems.SetAllMsgNum(Value: integer);
////////////////////////////////////////////////////////////////////////////////
// Set all items MsgNum property
var
  i : integer;
begin
  for i := 0 to Count-1 do
    Items[i].MsgNum := Value;
end;

procedure TMailItems.SetAllNew(Value: boolean);
////////////////////////////////////////////////////////////////////////////////
// Set all items New property
var
  i : integer;
begin
  for i := 0 to Count-1 do
    Items[i].New := Value;
end;

procedure TMailItems.DeleteAllMsgNum(MsgNum : integer);
////////////////////////////////////////////////////////////////////////////////
// Delete all items with the specified MsgNum
var
  i : integer;
begin
  i := 0;
  while i <= Count-1 do
  begin
    if Items[i].MsgNum = -1 then
      Self.Delete(i)
    else
      Inc(i);
  end;
end;

{ TAccountItem }

constructor TAccountItem.Create(Collection: TCollection);
begin
  inherited;
  UIDLSupported := True;
  UnknownMessages := nil;
end;

destructor TAccountItem.Destroy;
var Q: TUnknownMsg;
begin
  while (UnknownMessages <> nil) do
  begin
    Q := UnknownMessages;
    UnknownMessages := UnknownMessages^.Next;
    Dispose(Q);
  end;
  inherited;
end;

procedure TAccountItem.AssignTo(Dest: TPersistent);
begin
  if Dest is TAccountItem then
    with TAccountItem(Dest) do
    begin
      Name := self.Name;
      Server := self.Server;
      Port := self.Port;
      Protocol := self.Protocol;
      Login := self.Login;
      Password := self.Password;
      MailProgram := self.MailProgram;
      Sound := self.Sound;
      Color := self.Color;
      MsgIDs := self.MsgIDs;
      ViewedMsgIDs := self.ViewedMsgIDs;
      IgnoreCount := self.IgnoreCount;
      Enabled := self.Enabled;
      Error := self.Error;
      Status := self.Status;
      Interval := self.Interval;
      Timer := self.Timer;
      Size := self.Size;
      Connecting := self.Connecting;
      OldNum := self.OldNum;
      Prot := self.Prot;
      UIDLSupported := self.UIDLSupported;
      Mail := self.Mail;
    end
  else inherited AssignTo(Dest);
end;

{ TAccountItems }

constructor TAccountItems.Create;
begin
  inherited Create(TAccountItem);
end;

function TAccountItems.Add: TAccountItem;
begin
  Result := TAccountItem(inherited Add);
end;

function TAccountItems.GetItem(Index: Integer): TAccountItem;
begin
  Result := TAccountItem(inherited GetItem(Index));
end;

procedure TAccountItems.SetItem(Index: Integer; Value: TAccountItem);
begin
  inherited SetItem(Index, Value);
end;

procedure TAccountItems.Move(CurIndex, NewIndex: Integer);
var
  TempItem: TAccountItem;
begin
  if CurIndex <> NewIndex then
  begin
    BeginUpdate;
    try
      TempItem := TAccountItem.Create(nil);
      try
        TempItem.Assign(GetItem(CurIndex));
        Delete(CurIndex);
        Insert(NewIndex);
        SetItem(NewIndex,TempItem);
      finally
        TempItem.Free;
      end;
    finally
      EndUpdate;
    end;
  end;
end;

{ TRuleItem }

procedure TRuleItem.AssignTo(Dest: TPersistent);
begin
  if Dest is TRuleItem then
    with TRuleItem(Dest) do
    begin
      Name := self.Name;
      Enabled := self.Enabled;
      New := self.New;
      Account := self.Account;
      Area := self.Area;
      Compare := self.Compare;
      Text := self.Text;
      Wav := self.Wav;
      Delete := self.Delete;
      Ignore := self.Ignore;
      EXE := self.EXE;
      Important := self.Important;
      Spam := self.Spam;
      Protect := self.Protect;
      Log := self.Log;
    end
  else inherited AssignTo(Dest);
end;

{ TRuleItems }

constructor TRuleItems.Create;
begin
  inherited Create(TRuleItem);
end;

function TRuleItems.Add: TRuleItem;
begin
  Result := TRuleItem(inherited Add);
end;

function TRuleItems.GetItem(Index: Integer): TRuleItem;
begin
  Result := TRuleItem(inherited GetItem(Index));
end;

procedure TRuleItems.SetItem(Index: Integer; Value: TRuleItem);
begin
  inherited SetItem(Index, Value);
end;

procedure TRuleItems.Move(CurIndex, NewIndex: Integer);
var
  TempItem: TRuleItem;
begin
  if CurIndex <> NewIndex then
  begin
    BeginUpdate;
    try
      TempItem := TRuleItem.Create(nil);
      try
        TempItem.Assign(GetItem(CurIndex));
        Delete(CurIndex);
        Insert(NewIndex);
        SetItem(NewIndex,TempItem);
      finally
        TempItem.Free;
      end;
    finally
      EndUpdate;
    end;
  end;
end;


end.
